<?php
// scripts/sender.php
// Clean, simplified version of the email sender

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Logger.php';
require_once __DIR__ . '/../core/CsvHandler.php';
require_once __DIR__ . '/../core/SmtpSender.php';

Logger::info("--- START SENDER ---");

// 1. Lock and Read Send Queue
$lockedHandle = fopen(FILE_SEND_QUEUE, 'c+');
if (!flock($lockedHandle, LOCK_EX)) {
    Logger::error("Impossible de locker la queue pour envoi.");
    exit(1);
}

// Read content
rewind($lockedHandle);
$headers = fgetcsv($lockedHandle);
$rows = [];
if ($headers) {
    while (($r = fgetcsv($lockedHandle)) !== false) {
        if (count($r) == count($headers))
            $rows[] = array_combine($headers, $r);
    }
}

if (empty($rows)) {
    Logger::info("Send Queue vide.");
    flock($lockedHandle, LOCK_UN);
    fclose($lockedHandle);
    exit(0);
}

// 2. Take batch to process
$batch = array_slice($rows, 0, SMTP_MAX_MAILS_PER_MINUTE);
$remaining = array_slice($rows, SMTP_MAX_MAILS_PER_MINUTE);

// 3. Rewrite file immediately with remaining items
ftruncate($lockedHandle, 0);
rewind($lockedHandle);
fputcsv($lockedHandle, $headers);
foreach ($remaining as $r) {
    fputcsv($lockedHandle, $r);
}
fflush($lockedHandle);
flock($lockedHandle, LOCK_UN);
fclose($lockedHandle);

Logger::info("Locked and removed " . count($batch) . " items from queue.");

// 4. Now send emails
$smtp = new SmtpSender();
$sentCount = 0;

foreach ($batch as $item) {
    Logger::info("Envoi à {$item['email']} (UID: {$item['id']})");

    $sent = $smtp->send($item['email'], "Re: " . $item['subject'], $item['grok_response']);

    if ($sent) {
        $sentCount++;
        Logger::info("✅ Envoyé avec succès à {$item['email']}");

        // Archive
        $archiveRow = [
            'id' => $item['id'],
            'email' => $item['email'],
            'subject' => $item['subject'],
            'date_sent' => date('Y-m-d H:i:s'),
            'tag_zimbra' => $item['tag_zimbra'] ?? '',
            'final_response' => $item['grok_response'],
            'shop_id' => $item['shop_id'] ?? ''
        ];
        CsvHandler::append(FILE_ARCHIVES, $archiveRow);

        // Pause between mails
        if ($sentCount < count($batch)) {
            Logger::info("Pause de " . SMTP_PAUSE_BETWEEN_MAILS . "s...");
            sleep(SMTP_PAUSE_BETWEEN_MAILS);
        }
    } else {
        Logger::error("❌ Echec envoi à {$item['email']}, remise en queue.");
        // Re-append to queue for retry
        CsvHandler::append(FILE_SEND_QUEUE, $item);
    }
}

Logger::info("--- END SENDER ($sentCount envoyés) ---");
