<?php
// scripts/scan_loop.php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Logger.php';

// Strategy: 1 mail every 2 seconds.
// Run for approx 20 minutes max (600 iterations * 2s = 1200s).
$MAX_ITERATIONS = 600;
$BATCH_SIZE = 5;
$PAUSE_SECONDS = 2;

$LOCK_FILE = sys_get_temp_dir() . '/scan_loop.lock';
$MEMORY_LIMIT_MB = 64; // Lowered to 64MB for safety

// Priority: Very low
if (function_exists('posix_setpriority')) {
    posix_setpriority(PRIO_PROCESS, 0, 19);
}

if (file_exists($LOCK_FILE)) {
    // If older than 20 mins, delete.
    if (time() - filemtime($LOCK_FILE) > 1200) {
        unlink($LOCK_FILE);
    } else {
        // Robust check: Is the process actually running?
        $pid = trim(file_get_contents($LOCK_FILE));
        exec("kill -0 $pid 2>/dev/null", $outputKill, $returnKill);
        if ($pid && $returnKill === 0) {
            Logger::info("[LOOP] Lock exists and PID $pid is active. Exiting.");
            exit;
        } else {
            Logger::info("[LOOP] Stale lock file. Unlinking.");
            unlink($LOCK_FILE);
        }
    }
}

file_put_contents($LOCK_FILE, getmypid());
register_shutdown_function(function () use ($LOCK_FILE) {
    if (file_exists($LOCK_FILE))
        unlink($LOCK_FILE);
});

Logger::info("[LOOP] Starting Ultra-Safe Continuous Scan. Max $MAX_ITERATIONS.");

for ($i = 0; $i < $MAX_ITERATIONS; $i++) {
    // 1. MEMORY CHECK
    $currentMem = memory_get_usage(true) / 1024 / 1024;
    if ($currentMem > $MEMORY_LIMIT_MB) {
        Logger::error("[LOOP] Memory limit exceeded ($currentMem MB). Emergency stop.");
        break;
    }

    // 2. LOAD CHECK (Strict Throttling & Stop)
    $load = sys_getloadavg();
    $currentPause = $PAUSE_SECONDS;

    if ($load[0] > 1.0) { // If 1min load > 1.0
        Logger::error("[LOOP] Server Overload Detected (Load: {$load[0]}). STOPPING to protect SSH.");
        break;
    }

    if ($load[0] > 0.5) { // If 1min load > 0.5
        $currentPause = 15; // Extreme Throttling
        Logger::info("[LOOP] High load detected ($load[0]). Extreme throttling (Pause: 15s).");
    }

    $output = [];
    $returnVar = 0;

    // Call scan.php with nice -n 19
    exec("nice -n 19 php " . __DIR__ . "/scan.php $BATCH_SIZE 2>&1", $output, $returnVar);

    $outStr = implode("\n", $output);

    if (preg_match('/0 nouveaux mails/i', $outStr)) {
        Logger::info("[LOOP] Empty queue. Stopping.");
        break;
    }

    if ($returnVar !== 0) {
        Logger::error("[LOOP] Error: " . substr($outStr, 0, 100));
        break;
    }

    if ($i % 10 == 0) {
        Logger::info("[LOOP] Processed " . ($i + 1) . " mails... (Mem: " . round($currentMem, 1) . "MB)");
    }

    sleep($currentPause);
}

Logger::info("[LOOP] Finished.");
