<?php
// scripts/recheck_clients.php
// Re-checks all messages in queue.csv to see if they belong to existing clients

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Logger.php';
require_once __DIR__ . '/../core/CsvHandler.php';
require_once __DIR__ . '/../core/PrestaShop.php';

// Disable timeout
set_time_limit(0);

echo "🚀 Démarrage de la vérification clients...\n";

// Use CsvHandler to read (handles multiline correctly)
$queue = CsvHandler::read(FILE_QUEUE);
if (empty($queue)) {
    echo "Files d'attente vide.\n";
    exit(0);
}

echo "📋 " . count($queue) . " messages à analyser.\n";

$presta = new PrestaShop();
$updatedCount = 0;
$newQueue = [];

$i = 0;
foreach ($queue as $item) {
    $i++;
    $email = $item['email'];
    $currentTag = $item['tag_zimbra'] ?? '';

    echo "[$i/" . count($queue) . "] Analyse $email (Tag: $currentTag)... ";

    // Check if already client
    if ($currentTag === 'Yellow') {
        echo "Déjà OK (Jaune).\n";
        $newQueue[] = $item;
        continue;
    }

    // Check PrestaShop
    $clientInfo = $presta->searchClient($email);

    if ($clientInfo) {
        echo "✅ TROUVÉ sur " . $clientInfo['shop_name'] . "!\n";

        // Update tag if it wasn't a specific high priority tag (or just force it for now/User request)
        // User asked to "check if client and change tag".
        // We preserve "Blue" (Livraison) logic if we want, but user seems to want to see clients.
        // Let's set it to Yellow unless it's Blue/Red/Green. 
        // Actually, if it's "Purple" or "Default", we switch to Yellow.

        $shouldUpdate = ($currentTag === 'Purple' || $currentTag === 'Default' || $currentTag === '' || $currentTag === 'SPAM_SUSPECT');

        if ($shouldUpdate) {
            $item['tag_zimbra'] = 'Yellow';
            $updatedCount++;
        }

        // Always update Shop ID / Client ID info
        $item['shop_id'] = $clientInfo['shop_id'];
        $item['client_id'] = $clientInfo['customer']['id'];

    } else {
        echo "⚪ Non trouvé.\n";
    }

    $newQueue[] = $item;

    // Small pause to be nice to API? 
    // No, curl is distinct. 
}

echo "\n🔄 Sauvegarde des modifications...\n";

if ($updatedCount > 0) {
    // Backup
    copy(FILE_QUEUE, FILE_QUEUE . '.bak');

    // Write using CsvHandler logic (manual here for control)
    $fp = fopen(FILE_QUEUE, 'w');
    if ($fp) {
        fputcsv($fp, array_keys($newQueue[0]));
        foreach ($newQueue as $row) {
            fputcsv($fp, $row);
        }
        fclose($fp);
        echo "✅ $updatedCount messages mis à jour !\n";
    } else {
        echo "❌ Erreur écriture fichier.\n";
    }
} else {
    echo "ℹ️ Aucun changement de tag nécessaire (mais IDs mis à jour si trouvés).\n";
    // Force write anyway to save shop_ids if any were missing
    $fp = fopen(FILE_QUEUE, 'w');
    if ($fp) {
        fputcsv($fp, array_keys($newQueue[0]));
        foreach ($newQueue as $row) {
            fputcsv($fp, $row);
        }
        fclose($fp);
    }
}

echo "🏁 Terminé.\n";
