<?php
// scripts/recheck_clients.php
// Re-checks all messages in queue.csv to see if they belong to existing clients
// Updates tag to 'Yellow' if found.

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Logger.php';
require_once __DIR__ . '/../core/CsvHandler.php';
require_once __DIR__ . '/../core/PrestaShop.php';

Logger::info("--- START CLIENT RE-CHECK ---");

$queue = CsvHandler::read(FILE_QUEUE);
if (empty($queue)) {
    Logger::info("Queue vide. Fin.");
    exit(0);
}

$presta = new PrestaShop();
$updatedCount = 0;
$newQueue = [];

foreach ($queue as $item) {
    // Only check if not already identified as client or specific critical tags
    // If tag is already Yellow, skip.
    // If tag is Red (Cmd) or Blue (Livraison), maybe we keep it? 
    // User said "changer couleur des tags si client".
    // Usually "Client" (Yellow) is lower priority than "Problème Livraison" (Blue).
    // But if it's currently "Purple" (Unknown) or "Default", we definitely check.

    $currentTag = $item['tag_zimbra'] ?? '';

    // We re-check everyone who isn't already Yellow, just in case.
    // unlikely to overwrite stronger tags like Blue/Red if the AI decided Red.
    // But if the AI missed the client status, we should check.

    // Let's focus on 'Purple', 'Default', '' for now to be safe, 
    // OR just check everyone and if client found, ensure tag reflects it 
    // UNLESS intentions are stronger? 
    // Scan.php logic: Blue > Red > Green > Yellow > Purple.
    // If it's Blue, it remains Blue even if client.
    // If it's Purple, it becomes Yellow.

    // Initialize column if missing so CSV header is consistent
    if (!isset($item['last_order_date'])) {
        $item['last_order_date'] = '';
        $updatedCount++; // Force write to update CSV structure
    }

    if ($currentTag === 'Yellow') {
        $newQueue[] = $item;
        continue;
    }

    // Check PrestaShop
    $email = $item['email'];
    Logger::info("Checking $email...");
    $clientInfo = $presta->searchClient($email);

    if ($clientInfo) {
        Logger::info("✅ FOUND CLIENT for $email on {$clientInfo['shop_name']}");

        // If current tag is Purple or empty, promote to Yellow
        if ($currentTag === 'Purple' || $currentTag === 'Default' || $currentTag === '' || $currentTag === 'SPAM_SUSPECT') {
            $item['tag_zimbra'] = 'Yellow'; // TAG_CLIENT_RECONNU
            $item['shop_id'] = $clientInfo['shop_id'];
            $item['client_id'] = $clientInfo['customer']['id'];

            // Fetch Last Order Date
            $lastOrderDate = $presta->getLastOrderDate($clientInfo['shop_id'], $clientInfo['customer']['id']);
            if ($lastOrderDate) {
                $item['last_order_date'] = $lastOrderDate;
                Logger::info("  📅 Last Order: $lastOrderDate");
            }

            // We might want to update the grok_response to be more personalized? 
            // Too complex for a patch script. Just update tag.
            $updatedCount++;
        } else {
            Logger::info("Customer found but keeping existing high-priority tag: $currentTag");
            // Still update IDs
            $item['shop_id'] = $clientInfo['shop_id'];
            $item['client_id'] = $clientInfo['customer']['id'];
            $updatedCount++; // Technical update
        }
    } else {
        Logger::info("⚪ Not a client: $email");
    }

    $newQueue[] = $item;
}

if ($updatedCount > 0) {
    // Save back to CSV
    // We use overwrite mode safely here as this is a maintenance script
    // NOTE: If new emails came in during this runs (unlikely if cron off), they are safe if we use append? 
    // No, we read all, processing, writing all. 
    // Ideally we should lock. CsvHandler doesn't expose lock easily for overwrite.
    // But we are in maintenance mode.

    // Backup first
    copy(FILE_QUEUE, FILE_QUEUE . '.bak');

    $handle = fopen(FILE_QUEUE, 'w');
    if ($handle) {
        // Write header
        if (!empty($newQueue)) {
            fputcsv($handle, array_keys($newQueue[0]));
            foreach ($newQueue as $row) {
                fputcsv($handle, $row);
            }
        }
        fclose($handle);
        Logger::info("Mise à jour terminée. $updatedCount lignes modifiées.");
    } else {
        Logger::error("Impossible d'écrire dans queue.csv");
    }
} else {
    Logger::info("Aucun changement nécessaire.");
}

Logger::info("--- END CLIENT RE-CHECK ---");
