<?php
// core/PrestaShop.php

require_once __DIR__ . '/Logger.php';

class PrestaShop
{
    private $shops;

    public function __construct()
    {
        $creds = require __DIR__ . '/../config/credentials.php';
        $this->shops = $creds['prestashop'];
    }

    /**
     * Cherche un client par email sur toutes les boutiques.
     * @param string $email
     * @return array|null Info client + shop_id ou null
     */
    public function searchClient(string $email)
    {
        foreach ($this->shops as $id => $shop) {
            $url = rtrim($shop['url'], '/') . '/api/customers';
            $key = $shop['key'];

            // Construction URL avec filtre
            $queryUrl = $url . "?filter[email]=" . urlencode($email) . "&display=[id,firstname,lastname,email]&output_format=JSON";

            // Contexte HTTP pour Auth Basic
            $context = stream_context_create([
                'http' => [
                    'header' => "Authorization: Basic " . base64_encode($key . ":")
                ]
            ]);

            try {
                $response = @file_get_contents($queryUrl, false, $context);

                if ($response === false) {
                    Logger::debug("PrestaShop $id ({$shop['name']}) : Pas de réponse ou erreur.");
                    continue;
                }

                $json = json_decode($response, true);

                if (isset($json['customers']) && !empty($json['customers'])) {
                    // Client trouvé !
                    $customer = $json['customers'][0];
                    Logger::info("Client trouvé sur Shop $id ({$shop['name']}) : ID " . $customer['id']);

                    return [
                        'shop_id' => $id,
                        'shop_name' => $shop['name'],
                        'customer' => $customer
                    ];
                }

            } catch (Exception $e) {
                Logger::error("Erreur PrestaShop $id : " . $e->getMessage());
            }

            // Anti-DDOS Delay (1s) to avoid 429 errors
            sleep(1);
        }

        return null;
    }

    /**
     * Récupère la date de la dernière commande d'un client.
     * @param int $shopId
     * @param int $customerId
     * @return string|null Date YYYY-MM-DD HH:MM:SS ou null
     */
    public function getLastOrderDate(int $shopId, int $customerId)
    {
        if (!isset($this->shops[$shopId])) {
            return null;
        }

        $shop = $this->shops[$shopId];
        $url = rtrim($shop['url'], '/') . '/api/orders';
        $key = $shop['key'];

        // Filter by customer ID, sort by date desc, limit 1
        $queryUrl = $url . "?filter[id_customer]=$customerId&display=[date_add]&sort=[date_add_DESC]&limit=1&output_format=JSON";

        $context = stream_context_create([
            'http' => [
                'header' => "Authorization: Basic " . base64_encode($key . ":")
            ]
        ]);

        try {
            $response = @file_get_contents($queryUrl, false, $context);
            if ($response !== false) {
                $json = json_decode($response, true);
                if (isset($json['orders']) && !empty($json['orders'])) {
                    return $json['orders'][0]['date_add'];
                }
            }
        } catch (Exception $e) {
            Logger::error("Erreur getLastOrderDate Shop $shopId : " . $e->getMessage());
        }

        return null;
    }
}
