<?php
// core/ImapClient.php

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/Logger.php';

class ImapClient
{
    private $mbox;
    private $email;
    private $password;
    private $host;

    public function __construct()
    {
        $creds = include __DIR__ . '/../config/credentials.php';
        $this->email = $creds['email']['address'];
        $this->password = $creds['email']['password'];
        $this->host = $creds['email']['imap_host'];

        $this->connect();
    }

    private function connect()
    {
        // Connexion SSL
        $connectionString = "{" . $this->host . ":993/imap/ssl}INBOX";

        try {
            $this->mbox = imap_open($connectionString, $this->email, $this->password);

            if (!$this->mbox) {
                Logger::error("IMAP Connection failed: " . imap_last_error());
                throw new Exception("IMAP Connection failed");
            }
        } catch (Exception $e) {
            Logger::error("IMAP Critical Error: " . $e->getMessage());
        }
    }

    public function fetchUnread($limit = 10, $excludeIds = [])
    {
        if (!$this->mbox)
            return [];

        // 1. Get ALL Unread UIDs
        $uids = imap_search($this->mbox, 'UNSEEN', SE_UID);
        if (!$uids)
            return [];

        // Filter exclusions
        if (!empty($excludeIds)) {
            $uids = array_diff($uids, $excludeIds);
            if (empty($uids))
                return [];
        }

        // 2. Heuristic: Take last 300 UIDs (likely newest) to parse timestamps
        //    (Avoids parsing thousands of headers)
        rsort($uids); // High to Low
        $candidate_uids = array_slice($uids, 0, 300);

        // 3. Fetch Headers & Sort by Date
        $candidates = [];
        foreach ($candidate_uids as $uid) {
            $msgNo = imap_msgno($this->mbox, $uid);
            if (!$msgNo)
                continue;

            $header = imap_headerinfo($this->mbox, $msgNo);
            if (!$header)
                continue;

            $candidates[] = [
                'uid' => $uid,
                'msgno' => $msgNo,
                'udate' => $header->udate ?? 0,
                'header' => $header
            ];
        }

        // Real Sort: Timestamp Descending (Newest real time first)
        usort($candidates, function ($a, $b) {
            return $b['udate'] <=> $a['udate'];
        });

        // 4. Slice strictly the Limit
        $final_selection = array_slice($candidates, 0, $limit);

        // 5. Fetch Bodies for the winners
        $results = [];
        foreach ($final_selection as $item) {
            // SAFETY THROTTLE: Sleep 0.2s between fetches
            usleep(200000);

            $uid = $item['uid'];
            $msgNo = $item['msgno'];
            $header = $item['header'];

            $subject = $header->subject ?? '(No Subject)';
            $subject = mb_decode_mimeheader($subject); // Decode subject properly

            $fromStruct = $header->from[0];
            $mailbox = $fromStruct->mailbox;
            $host = $fromStruct->host;
            $emailAddr = $mailbox . "@" . $host;
            $personal = isset($fromStruct->personal) ? imap_utf8($fromStruct->personal) : '';

            // REVERT: 'from' is just email address for Dashboard Display
            $from = $emailAddr;

            $date = $header->date;

            // Body PEEK
            $body = imap_fetchbody($this->mbox, $msgNo, 1, FT_PEEK);
            if (!$body) {
                $structure = imap_fetchstructure($this->mbox, $msgNo);
                if (isset($structure->parts) && count($structure->parts)) {
                    $body = imap_fetchbody($this->mbox, $msgNo, 1.1, FT_PEEK);
                    if (!$body)
                        $body = imap_fetchbody($this->mbox, $msgNo, 1, FT_PEEK);
                }
            }

            $results[] = [
                'uid' => $uid,
                'subject' => $subject,
                'from' => $from,          // Clean Email
                'from_name' => $personal, // Separate Name for Spam Check
                'date' => $date,
                'timestamp' => $item['udate'],
                'body' => $body
            ];
        }

        return $results;
    }

    /**
     * Applique un tag Zimbra avec pause de sécurité STRICTE (30s AVANT et APRES).
     */
    public function setTag($msgId, $tag)
    {
        if (!$this->mbox)
            return false;

        // MAPPING ZIMBRA (To ensure visual colors)
        // Adjust these if client uses different mapping
        $map = [
            'Red' => '$Label1', // Rouge
            'Blue' => '$Label4', // Bleu ? Check Zimbra defs, often Label4=Blue
            'Green' => '$Label3', // Vert
            'Yellow' => '$Label2', // Orange/Jaune
            'Purple' => '$Label5', // Violet
            'Orange' => '$Label2'
        ];

        $zimbraTag = $map[$tag] ?? $tag;

        $logMsg = "Tentative Tagging UID $msgId : '$tag' -> Keyword '$zimbraTag'";
        $this->logTagging($logMsg);

        // PAUSE AVANT (CRITIQUE ZIMBRA)
        $this->logTagging("PAUSE 30s (Avant)...");
        sleep(30);

        $success = imap_setflag_full($this->mbox, (string) $msgId, $zimbraTag);

        if ($success) {
            $this->logTagging("SUCCES Tag '$zimbraTag'.");
            // PAUSE APRES (CRITIQUE ZIMBRA)
            $this->logTagging("PAUSE 30s (Apres)...");
            sleep(30);
        } else {
            $this->logTagging("ECHEC Tag '$zimbraTag' : " . imap_last_error());
        }

        return $success;
    }

    private function logTagging($msg)
    {
        $file = LOGS_DIR . '/imap_tagging.log';
        $entry = date('Y-m-d H:i:s') . " " . $msg . PHP_EOL;
        file_put_contents($file, $entry, FILE_APPEND);
        // Also standard log
        Logger::info("[TAG] " . $msg);
    }

    /**
     * Supprime un tag Zimbra
     */
    public function clearTag($msgId, $tag)
    {
        if (!$this->mbox)
            return false;
        return imap_clearflag_full($this->mbox, (string) $msgId, $tag);
    }

    public function getBody($msgId)
    {
        if (!$this->mbox)
            return "";
        return imap_fetchbody($this->mbox, $msgId, 1, FT_PEEK);
    }

    public function getStream()
    {
        return $this->mbox;
    }

    public function markAsRead($msgId)
    {
        if (!$this->mbox)
            return false;
        return imap_setflag_full($this->mbox, (string) $msgId, "\\Seen", ST_UID);
    }

    public function markAsUnread($msgId)
    {
        if (!$this->mbox)
            return false;
        return imap_clearflag_full($this->mbox, (string) $msgId, "\\Seen", ST_UID);
    }

    public function moveToFolder($msgId, $folderName)
    {
        if (!$this->mbox)
            return false;

        // Format: INBOX/Payplug for Zimbra
        $targetFolder = $folderName;

        // Move the message
        $result = imap_mail_move($this->mbox, (string) $msgId, $targetFolder, CP_UID);

        if ($result) {
            // Expunge to apply the move
            imap_expunge($this->mbox);
            Logger::info("Moved message UID $msgId to folder $targetFolder");
        } else {
            Logger::error("Failed to move message UID $msgId to folder $targetFolder: " . imap_last_error());
        }

        return $result;
    }

    public function expunge()
    {
        if (!$this->mbox)
            return false;
        return imap_expunge($this->mbox);
    }

    public function getUnreadUids()
    {
        if (!$this->mbox)
            return [];
        // SE_UID returns UIDs instead of Message Numbers
        $uids = imap_search($this->mbox, 'UNSEEN', SE_UID);
        return $uids ? $uids : [];
    }

    public function delete($uid)
    {
        if (!$this->mbox)
            return false;
        $msgNo = imap_msgno($this->mbox, $uid);
        if (!$msgNo)
            return false;

        return imap_delete($this->mbox, $msgNo);
    }

    public function move($uid, $folder)
    {
        if (!$this->mbox)
            return false;
        $msgNo = imap_msgno($this->mbox, $uid);
        if (!$msgNo)
            return false;

        // Try standard move
        return imap_mail_move($this->mbox, (string) $msgNo, $folder);
    }



    public function close()
    {
        if ($this->mbox)
            imap_close($this->mbox);
    }
}
