<?php
// core/CsvHandler.php

class CsvHandler
{

    /**
     * Lit un fichier CSV complet.
     * @param string $filepath Chemin absolu du fichier
     * @return array Tableau de donnes
     */
    public static function read(string $filepath): array
    {
        if (!file_exists($filepath)) {
            return [];
        }

        $handle = fopen($filepath, 'r');
        if (!$handle)
            return [];

        $data = [];
        // Verrou partagé pour la lecture
        if (flock($handle, LOCK_SH)) {
            $headers = fgetcsv($handle);
            if ($headers) {
                while (($row = fgetcsv($handle)) !== false) {
                    // Robustness: pad if row is shorter than headers (happens if new columns added)
                    if (count($row) < count($headers)) {
                        $row = array_pad($row, count($headers), '');
                    }
                    // Associate if counts match
                    if (count($headers) === count($row)) {
                        $data[] = array_combine($headers, $row);
                    } else {
                        $data[] = $row;
                    }
                }
            }
            flock($handle, LOCK_UN);
        }
        fclose($handle);
        return $data;
    }

    /**
     * Ajoute une ligne à la fin du fichier CSV.
     * @param string $filepath
     * @param array $data Tableau associatif ou indexé (doit correspondre aux headers)
     */
    public static function append(string $filepath, array $data): bool
    {
        // Créer le fichier avec headers si inexistant (devrait être fait à l'init, mais sécurité)
        $isNew = !file_exists($filepath);
        if ($isNew) {
            // First creation must include headers
            $headers = array_keys($data);
            $headerLine = implode(',', $headers) . "\n";
            file_put_contents($filepath, $headerLine);
        }

        // Build CSV line manually to ensure consistent Linux newlines (\n)
        // Avoid fputcsv which can be inconsistent
        $line = '';
        $first = true;
        foreach ($data as $cell) {
            if (!$first)
                $line .= ',';
            // Basic escaping: quotes around value if it contains comma or quote
            $cell = (string) $cell;
            if (strpos($cell, ',') !== false || strpos($cell, '"') !== false || strpos($cell, "\n") !== false) {
                $cell = '"' . str_replace('"', '""', $cell) . '"';
            }
            $line .= $cell;
            $first = false;
        }
        $line .= "\n";

        // Append safe
        return (file_put_contents($filepath, $line, FILE_APPEND | LOCK_EX) !== false);
    }

    /**
     * Réécrit complètement le fichier (pour mise à jour ou suppression).
     * @param string $filepath
     * @param array $allData Tableau de tableaux (lignes)
     * @param array $headers En-têtes (requis si $allData est vide)
     */
    public static function overwrite(string $filepath, array $allData, array $headers = []): bool
    {
        $handle = fopen($filepath, 'w'); // 'w' tronque le fichier
        if (!$handle)
            return false;

        $success = false;
        if (flock($handle, LOCK_EX)) {
            // Déterminer les headers
            if (empty($headers) && !empty($allData)) {
                $headers = array_keys($allData[0]);
            }

            if (!empty($headers)) {
                fputcsv($handle, $headers);
            }

            foreach ($allData as $row) {
                fputcsv($handle, $row);
            }

            fflush($handle);
            flock($handle, LOCK_UN);
            $success = true;
        }
        fclose($handle);
        return $success;
    }
}
