<?php
// scripts/scan.php

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Logger.php';
require_once __DIR__ . '/../core/CsvHandler.php';
require_once __DIR__ . '/../core/ImapClient.php';
require_once __DIR__ . '/../core/PrestaShop.php';
require_once __DIR__ . '/../core/AiEngine.php';

Logger::info("--- START SCAN ---");

// 1. Charger UIDs déjà connus (Queue, Send Queue, Archives) pour éviter doublons
$allFiles = [FILE_QUEUE, FILE_SEND_QUEUE, FILE_ARCHIVES];
$existingUids = [];

foreach ($allFiles as $f) {
    if (file_exists($f)) {
        $rows = CsvHandler::read($f);
        if (!empty($rows)) {
            $existingUids = array_merge($existingUids, array_column($rows, 'id'));
        }
    }
}
$existingUids = array_unique(array_filter($existingUids));
Logger::info("UIDs déjà connus : " . count($existingUids));

// Priority: Very low
if (function_exists('posix_setpriority')) {
    posix_setpriority(PRIO_PROCESS, 0, 19);
}

// 2. Connexion services
try {
    $imap = new ImapClient();
    $presta = new PrestaShop();
    $ai = new AiEngine();

    // Limit configurable via CLI arg (default 40)
    // HARD MAXIMUM: Never exceed 40 messages to prevent server overload
    $requestedLimit = isset($argv[1]) ? (int) $argv[1] : 40;
    $limit = min($requestedLimit, 40); // Force max 40

    if ($requestedLimit > 40) {
        Logger::info("⚠️ Requested limit $requestedLimit exceeds maximum. Capped to 40.");
    }

    // 3. Fetch Unread (PEEK)
    // Pass existingUids to ignore already processed mails (Read-Only Mode)
    $emails = $imap->fetchUnread($limit, $existingUids);
    Logger::info(count($emails) . " nouveaux mails (Limit $limit).");
} catch (Exception $e) {
    Logger::error("Init failed: " . $e->getMessage());
    if (isset($imap))
        $imap->close();
    exit(1);
}

try {
    foreach ($emails as $mail) {
        $uid = $mail['uid'];
        $from = $mail['from'];
        $fromName = $mail['from_name'] ?? ''; // New Field
        $subject = $mail['subject'];
        $date = $mail['date']; // "Fri, 2 Jan 2026 ..."
        $body = $mail['body'];

        Logger::info("Traitement Mail UID: $uid | De: $from | Nom: $fromName");

        // Double check (should be handled by fetchUnread but safe)
        if (in_array($uid, $existingUids))
            continue;

        $emailAddr = strtolower(trim($from));

        // --- 0. FILTRE ANTI-SPAM (Mots-clés) ---
        if (isSpamByKeyword($subject, $body, $emailAddr)) {
            Logger::info(">> SPAM DETECTED (Keyword). Ajout Queue 'SPAM_KEYWORD'.");
            // READ-ONLY: On ne marque PAS lu, on ajoute juste au CSV pour ignorer au prochain scan.
            $newRow = [
                'id' => $uid,
                'email' => $emailAddr,
                'subject' => $subject,
                'date_received' => date('Y-m-d H:i:s', $mail['timestamp']),
                'tag_zimbra' => 'Purple',
                'grok_response' => 'SPAM KEYWORD BLOCK',
                'status' => 'SPAM_KEYWORD', // Isolated from Suspicion IA
                'shop_id' => '',
                'client_id' => '',
                'body' => substr($body, 0, 5000),
                'sender_name' => $fromName
            ];
            CsvHandler::append(FILE_QUEUE, $newRow);
            continue;
        }

        // --- 0.5 FILTRE PAYPLUG (Messages automatiques) ---
        if (stripos($from, 'support@payplug.com') !== false || stripos($body, 'support@payplug.com') !== false) {
            Logger::info(">> PAYPLUG AUTO-MESSAGE. Ajout Queue 'PAYPLUG_AUTO'.");

            // Mark as read in Zimbra (no response needed for automatic messages)
            $imap->markAsRead($uid);
            Logger::info(">> Marked Payplug UID $uid as READ in Zimbra.");

            $newRow = [
                'id' => $uid,
                'email' => $emailAddr,
                'subject' => $subject,
                'date_received' => date('Y-m-d H:i:s', $mail['timestamp']),
                'tag_zimbra' => 'PayplugSystem',
                'grok_response' => '(Message automatique - Aucune réponse nécessaire)',
                'status' => 'PAYPLUG_AUTO',
                'shop_id' => '',
                'client_id' => '',
                'body' => substr($body, 0, 5000),
                'sender_name' => $fromName
            ];
            CsvHandler::append(FILE_QUEUE, $newRow);
            continue;
        }

        // A. Identification PrestaShop
        $clientInfo = $presta->searchClient($emailAddr);
        $isClient = ($clientInfo !== null);

        // B. Analyse I.A. (Intent)
        $intentTag = $ai->analyzeIntent($subject, $body);
        Logger::info(">> AI Intent: $intentTag");

        // GESTION SPAM IA
        if ($intentTag === 'Spam') {
            Logger::info(">> SPAM SUSPECTED BY AI.");
            // On ajoute à la queue avec statut SPAM_SUSPECT et Tag Purple (ou autre)
            $newRow = [
                'id' => $uid,
                'email' => $emailAddr,
                'subject' => $subject,
                'date_received' => date('Y-m-d H:i:s', $mail['timestamp']),
                'tag_zimbra' => 'Purple',
                'grok_response' => 'SPAM SUSPECTÉ PAR IA',
                'status' => 'SPAM_SUSPECT',
                'shop_id' => '',
                'client_id' => '',
                'body' => substr($body, 0, 5000), // Limit body size for CSV safety
                'sender_name' => $fromName
            ];
            CsvHandler::append(FILE_QUEUE, $newRow);
            // READ-ONLY: No setTag
            continue;
        }

        // C. Détermination Tag Final
        $finalTag = determineFinalTag($intentTag, $isClient);
        Logger::info(">> Final Tag: $finalTag");

        // D. Application Tag sur Zimbra
        // READ-ONLY: Disabled. Tags applied only via Dashboard "Force Re-Tag".
        /*
        if ($finalTag) {
            $imap->setTag($uid, $finalTag);
        }
        */

        // E. Génération Brouillon (Grok)
        $draft = $ai->generateDraft($subject, $body, $clientInfo);

        // F. Sauvegarde Queue CSV
        $newRow = [
            'id' => $uid,
            'email' => $emailAddr,
            'subject' => $subject,
            'date_received' => date('Y-m-d H:i:s', $mail['timestamp']),
            'tag_zimbra' => $finalTag,
            'grok_response' => $draft,
            'status' => 'PENDING',
            'shop_id' => $clientInfo ? $clientInfo['shop_id'] : '',
            'client_id' => $clientInfo ? $clientInfo['customer']['id'] : '',
            'body' => substr($body, 0, 5000),
            'sender_name' => $fromName
        ];

        // Final check: Load queue again to ensure no one added it in the meantime
        $freshQueue = CsvHandler::read(FILE_QUEUE);
        $freshUids = array_column($freshQueue, 'id');
        if (in_array($uid, $freshUids)) {
            Logger::info("UID $uid added by another process. Skipping.");
            continue;
        }

        if (CsvHandler::append(FILE_QUEUE, $newRow)) {
            Logger::info("Mail ajouté à la queue.");
            // Add to local list to avoid duplicates in SAME scan run
            $existingUids[] = $uid;
        } else {
            Logger::error("Erreur écriture CSV pour UID $uid");
        }
    }
} finally {
    if (isset($imap)) {
        $imap->close();
    }
}
Logger::info("--- END SCAN ---");


// --- Helpers ---

function determineFinalTag($intent, $isClient)
{
    // Tags Constants: TAG_CLIENT_RECONNU (Yellow), TAG_INTENTION_NOUVELLE (Red), etc.

    // Priorité Absolue : Problème Livraison
    if ($intent === 'Blue')
        return TAG_PROBLEME_LIVRAISON; // Blue

    // Priorité : Nouvelle Commande
    if ($intent === 'Red')
        return TAG_INTENTION_NOUVELLE; // Red

    // Priorité : Erreur
    if ($intent === 'Green')
        return TAG_INTENTION_ERREUR; // Green

    // Si Client Reconnu (et pas de problème majeur ci-dessus)
    if ($isClient)
        return TAG_CLIENT_RECONNU; // Yellow

    // Sinon
    if ($intent === 'Purple' || $intent === 'Unknown')
        return TAG_UNKNOWN; // Purple

    // Standard non client -> Purple ou rien ? Disons Purple pour arbitrage
    return TAG_UNKNOWN;
}

function isSpamByKeyword($subject, $body, $from)
{
    if (!file_exists(FILE_SPAM))
        return false;

    // Lecture CSV Spam simple
    // On peut utiliser CsvHandler mais pour perf, un fgetcsv rapide
    $handle = fopen(FILE_SPAM, 'r');
    if (!$handle)
        return false;

    // Skip headers
    fgetcsv($handle);

    $isSpam = false;
    $foundKeyword = '';

    $haystack = strtolower($subject . " " . $body . " " . $from);

    while (($row = fgetcsv($handle)) !== false) {
        $keyword = strtolower(trim($row[0]));
        if (empty($keyword))
            continue;

        if (strpos($haystack, $keyword) !== false) {
            $isSpam = true;
            $foundKeyword = $keyword;
            break;
        }
    }
    fclose($handle);

    if ($isSpam) {
        // Logging
        $logMsg = date('Y-m-d H:i:s') . " | BLOCK | $from | Keyword: $foundKeyword" . PHP_EOL;
        file_put_contents(FILE_SPAM_LOG, $logMsg, FILE_APPEND | LOCK_EX);
    }

    return $isSpam;
}
