<?php
// public/action.php

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/CsvHandler.php';
require_once __DIR__ . '/../core/ImapClient.php';
require_once __DIR__ . '/../core/AiEngine.php';
require_once __DIR__ . '/../core/PrestaShop.php';

header('Content-Type: application/json');

$action = $_GET['action'] ?? '';
$id = $_GET['id'] ?? '';

if (!$id && !in_array($action, ['refresh', 'expunge', 'get_spam_keywords', 'add_spam_keyword', 'remove_spam_keyword', 'trash_spam_keywords', 'preview_spam_matches', 'retag_queue', 'force_scan', 'generate_ai_draft', 'send_ai_draft', 'mark_as_read', 'mark_as_unread', 'get_facture_emails', 'add_facture_email', 'remove_facture_email', 'process_factures'])) {
    echo json_encode(['error' => 'Missing ID']);
    exit;
}

try {
    // Emergency stop
    if ($action === 'emergency_kill') {
        exec("sudo killall php");
        echo json_encode(['success' => true, 'message' => '💀 Tous les processus PHP ont été stoppés.']);
        exit;
    }

    // --- STANDARD ITEM ACTIONS ---
    if (in_array($action, ['accept', 'trash', 'reclassify'])) {
        $imap = new ImapClient();
        $queue = CsvHandler::read(FILE_QUEUE);
        $targetIndex = -1;
        $targetItem = null;

        foreach ($queue as $index => $item) {
            if ($item['id'] == $id) {
                $targetIndex = $index;
                $targetItem = $item;
                break;
            }
        }

        if ($targetIndex === -1) {
            echo json_encode(['error' => 'Item not found']);
            exit;
        }

        if ($action === 'accept') {
            CsvHandler::append(FILE_SEND_QUEUE, $targetItem);
            unset($queue[$targetIndex]);
            CsvHandler::overwrite(FILE_QUEUE, array_values($queue));
            $imap->markAsRead($id);
            echo json_encode(['success' => true, 'message' => 'Validé et mis en fichier d\'envoi.']);
        }

        if ($action === 'trash') {
            // Individual item manual deletion = Trash
            $imap->delete($id);
            $imap->expunge();
            unset($queue[$targetIndex]);
            CsvHandler::overwrite(FILE_QUEUE, array_values($queue));
            echo json_encode(['success' => true, 'message' => 'Supprimé de Zimbra et de la liste.']);
        }

        if ($action === 'reclassify') {
            $newTag = $_GET['new_tag'] ?? '';
            if (!$newTag)
                throw new Exception("Tag manquant");
            $imap->setTag($id, $newTag);
            $ai = new AiEngine();
            $body = imap_fetchbody($imap->getStream(), $id, 1, FT_PEEK);
            $draft = $ai->generateDraft($targetItem['subject'], $body, null);
            $queue[$targetIndex]['tag_zimbra'] = $newTag;
            $queue[$targetIndex]['grok_response'] = $draft;
            CsvHandler::overwrite(FILE_QUEUE, array_values($queue));
            echo json_encode(['success' => true, 'message' => 'Reclassifié et nouveau brouillon généré.']);
        }
        $imap->close();
        exit;
    }

    // --- ACTION EXPUNGE ---
    if ($action === 'expunge') {
        $imap = new ImapClient();
        Logger::info("Action Expunge demandée.");
        sleep(30);
        $success = $imap->expunge();
        $imap->close();
        if ($success) {
            sleep(30);
            echo json_encode(['success' => true, 'message' => 'Corbeille vidée.']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Erreur vidage corbeille.']);
        }
        exit;
    }

    // --- GET SPAM KEYWORDS ---
    if ($action === 'get_spam_keywords') {
        $keywords = [];
        if (file_exists(FILE_SPAM)) {
            $content = file_get_contents(FILE_SPAM);
            $lines = explode("\n", $content);
            array_shift($lines); // header
            foreach ($lines as $line) {
                $line = trim($line);
                if (empty($line))
                    continue;
                $data = str_getcsv($line);
                if (!empty($data[0])) {
                    $keywords[] = $data[0];
                }
            }
        }
        echo json_encode(['success' => true, 'keywords' => $keywords]);
        exit;
    }

    // --- ADD SPAM KEYWORD ---
    if ($action === 'add_spam_keyword') {
        $kw = trim($_POST['keyword'] ?? '');
        if (empty($kw)) {
            echo json_encode(['success' => false, 'message' => 'Mot-clé vide']);
            exit;
        }

        // Check duplicates robustness
        $content = file_exists(FILE_SPAM) ? file_get_contents(FILE_SPAM) : "keyword,date_added\n";
        if (stripos($content, $kw) !== false) {
            echo json_encode(['success' => false, 'message' => 'Mot-clé déjà existant']);
            exit;
        }

        $fp = fopen(FILE_SPAM, 'a');
        fputcsv($fp, [$kw, date('Y-m-d')]);
        fclose($fp);

        echo json_encode(['success' => true]);
        exit;
    }

    if ($action === 'remove_spam_keyword') {
        $word = trim($_POST['word'] ?? '');
        if (empty($word)) {
            echo json_encode(['success' => false]);
            exit;
        }
        $rows = [];
        if (file_exists(FILE_SPAM)) {
            $handle = fopen(FILE_SPAM, 'r');
            if ($handle) {
                $header = fgetcsv($handle);
                $rows[] = $header;
                while (($data = fgetcsv($handle)) !== false) {
                    if ($data[0] !== $word)
                        $rows[] = $data;
                }
                fclose($handle);
            }
        }
        CsvHandler::overwrite(FILE_SPAM, $rows);
        echo json_encode(['success' => true]);
        exit;
    }

    // --- CHECK SCAN STATUS ---
    if ($action === 'check_scan_status') {
        $lockFile = sys_get_temp_dir() . '/scan_loop.lock';
        $isRunning = false;
        if (file_exists($lockFile)) {
            if (time() - filemtime($lockFile) < 1200) { // 20 mins max validity
                $isRunning = true;
            }
        }
        echo json_encode(['success' => true, 'running' => $isRunning]);
        exit;
    }

    // --- ACTION FORCE SCAN (BACKGROUND LOOP) ---
    if ($action === 'force_scan') {
        $scriptPath = realpath(__DIR__ . '/../scripts/scan_loop.php');
        if (!file_exists($scriptPath)) {
            echo json_encode(['success' => false, 'message' => 'Script loop introuvable']);
            exit;
        }

        // Launch in background only if not already running
        $checkCmd = "pgrep -f scan_loop.php";
        exec($checkCmd, $outputCheck, $returnCheck);

        if ($returnCheck === 0) {
            echo json_encode(['success' => true, 'message' => '🚀 La synchronisation est déjà en cours d\'exécution.']);
            exit;
        }

        // nohup nice -n 19 ionice -c 3 php ... > /dev/null 2>&1 &
        $cmd = "nohup nice -n 19 ionice -c 3 php $scriptPath > /dev/null 2>&1 &";
        exec($cmd);

        echo json_encode([
            'success' => true,
            'message' => '🤖 Scan Automatique Démarré (30 mails toutes les 30s).\nVous pouvez fermer la fenêtre, ça tourne en fond.'
        ]);
        exit;
    }

    // --- PREVIEW SPAM ---
    if ($action === 'preview_spam_matches') {
        $keywords = [];
        if (file_exists(FILE_SPAM)) {
            $rows = array_map('str_getcsv', file(FILE_SPAM));
            array_shift($rows);
            foreach ($rows as $r) {
                if (!empty($r[0]))
                    $keywords[] = strtolower(trim($r[0]));
            }
        }

        $queue = CsvHandler::read(FILE_QUEUE);
        $matches = [];

        foreach ($queue as $item) {
            $subject = strtolower($item['subject'] ?? '');
            $email = strtolower($item['email'] ?? '');
            $senderName = strtolower($item['sender_name'] ?? '');
            $body = strtolower($item['body'] ?? '');

            $matchedKw = null;
            foreach ($keywords as $kw) {
                if (
                    strpos($subject, $kw) !== false ||
                    strpos($email, $kw) !== false ||
                    strpos($senderName, $kw) !== false ||
                    strpos($body, $kw) !== false ||
                    strpos(strtolower($item['grok_response'] ?? ''), $kw) !== false
                ) {
                    $matchedKw = $kw;
                    break;
                }
            }

            if ($matchedKw) {
                $matches[] = [
                    'id' => $item['id'],
                    'subject' => $item['subject'],
                    'from' => $item['email'],
                    'date' => $item['date_received'], // Added for sorting
                    'sender_name' => $item['sender_name'] ?? '',
                    'matched_keyword' => $matchedKw
                ];
            }
        }

        // Sort Matches by Date Desc
        usort($matches, function ($a, $b) {
            return strtotime($b['date']) <=> strtotime($a['date']);
        });


        echo json_encode([
            'success' => true,
            'matches' => $matches,
            'active_keywords' => $keywords
        ]);
        exit;
    }

    // --- TRASH SPAM (MOVE TO CORBEILLE) ---
    if ($action === 'trash_spam_keywords') {
        $imap = new ImapClient(); // Init Here
        $keywords = [];
        if (file_exists(FILE_SPAM)) {
            $rows = array_map('str_getcsv', file(FILE_SPAM));
            array_shift($rows);
            foreach ($rows as $r) {
                if (!empty($r[0]))
                    $keywords[] = strtolower(trim($r[0]));
            }
        }

        if (empty($keywords)) {
            echo json_encode(['success' => false, 'message' => "Aucun mot clé."]);
            exit;
        }

        $queue = CsvHandler::read(FILE_QUEUE);
        $deletedCount = 0;
        $idxToRemove = [];

        foreach ($queue as $index => $item) {
            $subject = strtolower($item['subject'] ?? '');
            $email = strtolower($item['email'] ?? '');
            $senderName = strtolower($item['sender_name'] ?? '');
            $body = strtolower($item['body'] ?? '');

            $isSpam = false;
            foreach ($keywords as $kw) {
                if (
                    strpos($subject, $kw) !== false ||
                    strpos($email, $kw) !== false ||
                    strpos($senderName, $kw) !== false ||
                    strpos($body, $kw) !== false
                ) {
                    $isSpam = true;
                    break;
                }
            }

            if ($isSpam) {
                Logger::info("Spam Clean: Moving UID " . $item['id'] . " to Trash.");
                try {
                    $success = $imap->move($item['id'], 'Trash');
                    if (!$success)
                        $success = $imap->move($item['id'], 'INBOX.Trash');
                    if (!$success)
                        $success = $imap->move($item['id'], 'Corbeille');

                    if ($success)
                        $deletedCount++;
                } catch (Exception $e) {
                    Logger::error("Failed move UID " . $item['id']);
                }
                $idxToRemove[] = $index;
            }
        }

        if (!empty($idxToRemove)) {
            try {
                $imap->expunge();
            } catch (Exception $e) {
            }
            foreach ($idxToRemove as $idx) {
                unset($queue[$idx]);
            }
            CsvHandler::overwrite(FILE_QUEUE, array_values($queue));
        }

        echo json_encode(['success' => true, 'message' => "Nettoyage terminé : $deletedCount mails déplacés vers Corbeille."]);
        $imap->close();
        exit;
    }

    // --- GENERATE AI DRAFT ---
    if ($action === 'generate_ai_draft') {
        $queue = CsvHandler::read(FILE_QUEUE);
        $targetIndex = -1;
        $targetItem = null;

        foreach ($queue as $index => $item) {
            if ($item['id'] == $id) {
                $targetIndex = $index;
                $targetItem = $item;
                break;
            }
        }

        if ($targetIndex === -1) {
            echo json_encode(['success' => false, 'error' => 'Item not found in queue']);
            exit;
        }

        $ai = new AiEngine();
        $presta = new PrestaShop();
        $clientParams = null;

        // Try to get client context if possible
        if (!empty($targetItem['email'])) {
            $clientParams = $presta->searchClient($targetItem['email']);
        }

        $draft = $ai->generateDraft($targetItem['subject'], $targetItem['body'], $clientParams);

        // Update persistence
        $queue[$targetIndex]['grok_response'] = $draft;
        CsvHandler::overwrite(FILE_QUEUE, array_values($queue));

        echo json_encode([
            'success' => true,
            'draft' => $draft
        ]);
        exit;
    }

    // --- SEND AI DRAFT ---
    if ($action === 'send_ai_draft') {
        $draft = $_POST['draft'] ?? '';
        if (empty($draft)) {
            echo json_encode(['success' => false, 'error' => 'Draft is required']);
            exit;
        }

        $queue = CsvHandler::read(FILE_QUEUE);
        $targetIndex = -1;
        $targetItem = null;

        foreach ($queue as $index => $item) {
            if ($item['id'] == $id) {
                $targetIndex = $index;
                $targetItem = $item;
                break;
            }
        }

        if ($targetIndex === -1) {
            echo json_encode(['success' => false, 'error' => 'Item not found in queue']);
            exit;
        }

        // Update the draft and add to send queue
        $targetItem['grok_response'] = $draft;
        CsvHandler::append(FILE_SEND_QUEUE, $targetItem);

        // Remove from main queue
        unset($queue[$targetIndex]);
        CsvHandler::overwrite(FILE_QUEUE, array_values($queue));

        // Mark as read in Zimbra
        $imap = new ImapClient();
        $imap->markAsRead($id);
        $imap->close();

        echo json_encode([
            'success' => true,
            'message' => 'Message ajouté à la file d\'envoi et marqué comme lu.'
        ]);
        exit;
    }

    // --- MARK AS READ ---
    if ($action === 'mark_as_read') {
        $imap = new ImapClient();
        $imap->markAsRead($id);

        // Check if this is a Payplug message and move it
        $queue = CsvHandler::read(FILE_QUEUE);
        foreach ($queue as $item) {
            if ($item['id'] == $id && ($item['status'] ?? '') === 'PAYPLUG_AUTO') {
                // Move to INBOX/Payplug folder
                $imap->moveToFolder($id, 'INBOX/Payplug');
                Logger::info("Moved Payplug message UID $id to INBOX/Payplug");
                break;
            }
        }

        $imap->close();

        echo json_encode([
            'success' => true,
            'message' => 'Message marqué comme lu.'
        ]);
        exit;
    }

    // --- MARK AS UNREAD ---
    if ($action === 'mark_as_unread') {
        $imap = new ImapClient();
        $imap->markAsUnread($id);
        $imap->close();

        echo json_encode([
            'success' => true,
            'message' => 'Message marqué comme non lu.'
        ]);
        exit;
    }

    // --- GET FACTURE EMAILS ---
    if ($action === 'get_facture_emails') {
        $emails = [];
        if (file_exists(FILE_FACTURES)) {
            // Force strict parsing
            $content = file_get_contents(FILE_FACTURES);
            $lines = explode("\n", $content);
            array_shift($lines); // Remove header

            foreach ($lines as $line) {
                $line = trim($line);
                if (empty($line))
                    continue;

                $data = str_getcsv($line);
                if (!empty($data[0])) {
                    $emails[] = $data[0];
                }
            }
        }
        echo json_encode(['success' => true, 'emails' => $emails]);
        exit;
    }

    // --- ADD FACTURE EMAIL ---
    if ($action === 'add_facture_email') {
        $email = trim($_POST['email'] ?? '');
        if (empty($email)) {
            echo json_encode(['success' => false, 'message' => 'Email vide']);
            exit;
        }

        // Clean email
        $email = filter_var($email, FILTER_SANITIZE_EMAIL);

        // Check duplicates
        $content = file_exists(FILE_FACTURES) ? file_get_contents(FILE_FACTURES) : "email,date_added\n";
        if (stripos($content, $email) !== false) {
            echo json_encode(['success' => false, 'message' => 'Email déjà existant']);
            exit;
        }

        // Append cleanly
        $fp = fopen(FILE_FACTURES, 'a');
        fputcsv($fp, [$email, date('Y-m-d')]);
        fclose($fp);

        echo json_encode(['success' => true]);
        exit;
    }

    // --- REMOVE FACTURE EMAIL ---
    if ($action === 'remove_facture_email') {
        $email = trim($_POST['email'] ?? '');
        if (empty($email)) {
            echo json_encode(['success' => false, 'message' => 'Email vide']);
            exit;
        }

        if (!file_exists(FILE_FACTURES)) {
            echo json_encode(['success' => false, 'message' => 'Fichier inexistant']);
            exit;
        }

        $rows = array_map('str_getcsv', file(FILE_FACTURES));
        $header = array_shift($rows);
        $filtered = array_filter($rows, fn($r) => $r[0] !== $email);

        // Rewrite file
        $fp = fopen(FILE_FACTURES, 'w');
        fputcsv($fp, $header);
        foreach ($filtered as $row) {
            fputcsv($fp, $row);
        }
        fclose($fp);

        echo json_encode(['success' => true]);
        exit;
    }

    // --- PROCESS FACTURES ---
    if ($action === 'process_factures') {
        // Get facture emails
        $factureEmails = [];
        if (file_exists(FILE_FACTURES)) {
            $rows = array_map('str_getcsv', file(FILE_FACTURES));
            array_shift($rows);
            foreach ($rows as $r) {
                if (!empty($r[0]))
                    $factureEmails[] = strtolower($r[0]);
            }
        }

        if (empty($factureEmails)) {
            echo json_encode(['success' => false, 'message' => 'Aucune adresse email enregistrée']);
            exit;
        }

        // Get unread messages and filter
        $imap = new ImapClient();
        $unreadUids = $imap->getUnreadUids();

        $processed = 0;
        $moved = 0;

        // Get queue to find matching emails
        // Get queue to find matching emails
        $queue = CsvHandler::read(FILE_QUEUE);
        $queueUpdated = false;

        foreach ($queue as &$item) {
            $uid = $item['id'];
            $fromEmail = strtolower($item['from'] ?? $item['email'] ?? '');

            // Check if this message is from a facture email
            $isFacture = false;
            foreach ($factureEmails as $factureEmail) {
                if (stripos($fromEmail, $factureEmail) !== false) {
                    $isFacture = true;
                    break;
                }
            }

            // If match found
            if ($isFacture) {
                Logger::info("DEBUG: Facture found in queue ($uid) from $fromEmail");

                // Check if UID is in unread list
                if (in_array($uid, $unreadUids)) {
                    $imap->markAsRead($uid);
                    $imap->moveToFolder($uid, 'inbox/Factures');
                    $processed++;
                    $moved++;
                    Logger::info("Processed facture UID $uid from $fromEmail");
                } else {
                    Logger::info("DEBUG: UID $uid skipped because NOT in unread list (Already read?)");
                }

                // CRITIQUE : Si le message était "En Attente" (PENDING) ou autre, on le marque comme TRAITÉ
                // pour qu'il disparaisse du compteur "En Attente"
                if (($item['status'] ?? '') !== 'FACTURE_DONE') {
                    $item['status'] = 'FACTURE_DONE';
                    $queueUpdated = true;
                }
            }
        }
        unset($item);

        // Save Queue Updates to clean up "Pending" list
        if ($queueUpdated) {
            CsvHandler::overwrite(FILE_QUEUE, $queue);
        }

        $imap->close();

        echo json_encode([
            'success' => true,
            'processed' => $processed,
            'moved' => $moved
        ]);
        exit;
    }

} catch (Exception $e) {
    Logger::error("Erreur API : " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
