<?php
// core/AiEngine.php

require_once __DIR__ . '/Logger.php';

class AiEngine
{
    private $keys;

    public function __construct()
    {
        $creds = require __DIR__ . '/../config/credentials.php';
        $this->keys = $creds['api_keys'];
    }

    /**
     * Analyse l'intention du mail pour déterminer le TAG couleur.
     * @return string Un des tags (Red, Green, Blue, Purple, Standard)
     */
    public function analyzeIntent($subject, $body)
    {
        $prompt = $this->getTriagePrompt($subject, $body);

        // 1. Essai Gemini (Gratuit & Rapide)
        try {
            return $this->callGemini($prompt, true); // true = expect short tag
        } catch (Exception $e) {
            Logger::error("Triage Gemini Failed: " . $e->getMessage() . " -> Switch OpenRouter Llama");
        }

        // 2. Essai OpenRouter (Llama)
        try {
            return $this->callOpenRouter($prompt, 'meta-llama/llama-3-70b-instruct', true);
        } catch (Exception $e) {
            Logger::error("Triage Llama Failed: " . $e->getMessage() . " -> Switch OpenRouter Qwen");
        }

        // 3. Essai OpenRouter (Qwen)
        try {
            return $this->callOpenRouter($prompt, 'qwen/qwen-2.5-72b-instruct', true);
        } catch (Exception $e) {
            Logger::error("Triage Qwen Failed -> Defaulting to Purple");
            return 'Purple';
        }
    }

    /**
     * Génère un brouillon de réponse.
     */
    public function generateDraft($subject, $body, $clientParams)
    {
        $prompt = $this->getDraftPrompt($subject, $body, $clientParams);
        $usedAI = 'unknown';

        // 1. Essai Grok (Premium - Qualité Max)
        try {
            $response = $this->callGrok($prompt);
            $usedAI = 'Grok';
            return $this->parseDraftResponse($response, $usedAI, $subject, $body);
        } catch (Exception $e) {
            Logger::error("Draft Grok Failed: " . $e->getMessage() . " -> Switch Gemini");
        }

        // 2. Fallback Gemini
        try {
            $response = $this->callGemini($prompt);
            $usedAI = 'Gemini';
            return $this->parseDraftResponse($response, $usedAI, $subject, $body);
        } catch (Exception $e) {
            Logger::error("Draft Gemini Failed -> Empty Draft");
            return "Erreur génération brouillon.";
        }
    }

    /**
     * Parse draft response to handle multilingual format
     * Returns JSON if translation present, otherwise plain text
     */
    private function parseDraftResponse($response, $usedAI = 'Grok', $subject = '', $body = '')
    {
        // Check for new format with question and response translations
        if (strpos($response, '---QUESTION_TRANSLATION---') !== false && strpos($response, '---RESPONSE_TRANSLATION---') !== false) {
            $parts = explode('---QUESTION_TRANSLATION---', $response, 2);
            $original = trim($parts[0]);

            $translationParts = explode('---RESPONSE_TRANSLATION---', $parts[1], 2);
            $questionTranslation = trim($translationParts[0]);
            $responseTranslation = trim($translationParts[1]);

            return json_encode([
                'type' => 'multilingual',
                'original' => $original,
                'question_translation' => $questionTranslation,
                'translation' => $responseTranslation,
                'ai_used' => 'Grok',
                'translation_ai' => 'Gemini'
            ], JSON_UNESCAPED_UNICODE);
        }

        // Legacy format check (backward compatibility)
        if (strpos($response, '---TRANSLATION---') !== false) {
            $parts = explode('---TRANSLATION---', $response, 2);
            $original = trim($parts[0]);
            $translation = trim($parts[1]);

            return json_encode([
                'type' => 'multilingual',
                'original' => $original,
                'translation' => $translation,
                'ai_used' => 'Grok',
                'translation_ai' => 'Gemini'
            ], JSON_UNESCAPED_UNICODE);
        }

        // French-only response (no translation needed)
        return $response;
    }

    // --- API CALLS ---

    private function callGemini($prompt, $isTriage = false)
    {
        $url = "https://generativelanguage.googleapis.com/v1beta/models/gemini-1.5-flash:generateContent?key=" . $this->keys['gemini'];

        $data = [
            "contents" => [
                [
                    "parts" => [["text" => $prompt]]
                ]
            ],
            "generationConfig" => [
                "temperature" => 0.3, // Basse temp pour triage
                "maxOutputTokens" => $isTriage ? 50 : 800
            ]
        ];

        $response = $this->httpPost($url, $data);
        if (!$response)
            throw new Exception("Empty response from Gemini");

        $json = json_decode($response, true);
        if (isset($json['candidates'][0]['content']['parts'][0]['text'])) {
            return trim($json['candidates'][0]['content']['parts'][0]['text']);
        }
        Logger::error("Gemini Raw Response: " . substr($response, 0, 500));
        throw new Exception("Invalid Gemini format");
    }

    private function callGrok($prompt)
    {
        // Grok via xAI API
        $url = "https://api.x.ai/v1/chat/completions";

        $data = [
            "model" => "grok-3", // Updated to grok-3 (grok-beta was deprecated)
            "messages" => [
                ["role" => "system", "content" => "Tu es un assistant de service client expert. Réponds en français, empathique et professionnel."],
                ["role" => "user", "content" => $prompt]
            ],
            "temperature" => 0.7
        ];

        $headers = ["Authorization: Bearer " . $this->keys['grok']];
        $response = $this->httpPost($url, $data, $headers);

        $json = json_decode($response, true);
        if (isset($json['choices'][0]['message']['content'])) {
            return trim($json['choices'][0]['message']['content']);
        }
        Logger::error("Grok Raw Response: " . substr($response, 0, 500));
        throw new Exception("Invalid Grok format");
    }

    private function callOpenRouter($prompt, $model, $isTriage = false)
    {
        $url = "https://openrouter.ai/api/v1/chat/completions";
        $key = $this->keys['openrouter']['main']; // Use main key for router

        $data = [
            "model" => $model,
            "messages" => [
                ["role" => "user", "content" => $prompt]
            ],
            "temperature" => 0.3
        ];

        $headers = ["Authorization: Bearer " . $key];
        $response = $this->httpPost($url, $data, $headers);

        $json = json_decode($response, true);
        if (isset($json['choices'][0]['message']['content'])) {
            return trim($json['choices'][0]['message']['content']);
        }
        throw new Exception("Invalid OpenRouter format");
    }

    private function httpPost($url, $data, $extraHeaders = [])
    {
        $ch = curl_init($url);
        $payload = json_encode($data);

        $headers = array_merge(['Content-Type: application/json'], $extraHeaders);

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $result = curl_exec($ch);
        if (curl_errno($ch)) {
            throw new Exception(curl_error($ch));
        }
        curl_close($ch);
        return $result;
    }

    // --- PROMPTS ---

    private function getTriagePrompt($subject, $body)
    {
        return "Analyse le mail suivant.
        Objet : $subject
        Corps : $body
        
        CLASSIFICATION STRICTE (Réponds UNIQUEMENT le mot clé) :
        - 'Red' : Nouvelle commande de plaque, demande de devis, 'je veux commander'.
        - 'Green' : Erreur sur la commande reçue (plaque cassée, erreur texte, mauvaise couleur).
        - 'Blue' : Problème de livraison, colis non reçu, retard, 'où est mon colis'.
        - 'Spam' : Publicité, arnaque, SEO, Viagra, Casino, Bitcoin, proposition commerciale non sollicitée.
        - 'Unknown' : Demande vague, incompréhensible ou hors sujet.
        - 'Standard' : Aucune des catégories ci-dessus (question simple, merci, etc).
        
        Réponse (un mot) :";
    }

    private function getDraftPrompt($subject, $body, $clientParams)
    {
        $clientInfo = $clientParams ? "Client identifié : " . $clientParams['customer']['firstname'] . " " . $clientParams['customer']['lastname'] . " (Client ID: {$clientParams['customer']['id']})" : "Client inconnu (prospect).";

        return "Tu es le service client de 'Plaque De Cadre et Constructeur'.
        
        1) Détecte la langue du message client
        2) Rédige une réponse courte, polie et serviable dans LA MÊME LANGUE que le client
        3) Si la langue N'EST PAS le français :
           - Ajoute : ---QUESTION_TRANSLATION---
           - Suivi de la traduction française de la question du client
           - Puis ajoute : ---RESPONSE_TRANSLATION---
           - Suivi de la traduction française de ta réponse
        
        NE mets PAS de placeholders type '[Votre Nom]'. Signe 'Le Service Client'.
        NE répète PAS les numéros d'étapes dans ta réponse.
        
        Contexte Client : $clientInfo
        Mail Client :
        Objet: $subject
        Corps: $body
        
        Réponse :";
    }
}
