<?php
// scripts/sender.php

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Logger.php';
require_once __DIR__ . '/../core/CsvHandler.php';
require_once __DIR__ . '/../core/SmtpSender.php';

Logger::info("--- START SENDER ---");

// 1. Lire Send Queue
$queue = CsvHandler::read(FILE_SEND_QUEUE);
if (empty($queue)) {
    Logger::info("Send Queue vide.");
    exit(0);
}

// 2. Sélectionner max 2 mails
$toProcess = array_slice($queue, 0, SMTP_MAX_MAILS_PER_MINUTE);
$remaining = array_slice($queue, SMTP_MAX_MAILS_PER_MINUTE);

$smtp = new SmtpSender();
$sentCount = 0;

foreach ($toProcess as $item) {
    Logger::info("Envoi mail à {$item['email']} (UID: {$item['id']})...");

    $success = $smtp->send($item['email'], "Re: " . $item['subject'], $item['grok_response']);

    if ($success) {
        $sentCount++;
        // Archiver
        $item['date_sent'] = date('Y-m-d H:i:s');
        $item['final_response'] = $item['grok_response']; // S'assurer que les clés correspondent
        $archiveRow = [
            'id' => $item['id'],
            'email' => $item['email'],
            'subject' => $item['subject'],
            'date_sent' => $item['date_sent'],
            'tag_zimbra' => $item['tag_zimbra'],
            'final_response' => $item['final_response'],
            'shop_id' => $item['shop_id'] ?? ''
        ];

        CsvHandler::append(FILE_ARCHIVES, $archiveRow);

        // Pause entre mails (anti-spam)
        if ($sentCount < count($toProcess)) {
            sleep(SMTP_PAUSE_BETWEEN_MAILS);
        }
    } else {
        Logger::error("Echec envoi à {$item['email']}. Remis en fin de file (ou à gérer manuellement).");
        // Stratégie simple : On le remet dans $remaining pour ne pas le perdre, ou on loggue erreur critique.
        // Pour l'instant, on le remet dans la queue processed ? Non, on risque de boucler.
        // On le laisse dans remaining pour retenter plus tard ?
        // Simplification : On le perd pas, on le remet à la fin.
        $remaining[] = $item;
    }
}

// 3. Réécrire la Send Queue (sans les envoyés)
// Attention : en production, il faudrait relire le fichier et faire un diff pour ne pas écraser 
// les ajouts faits PENDANT l'envoi (Dashboard -> Accept).
// Solution : Verrouillage Optimiste ou Diff strict.
// CsvHandler::overwrite écrase tout.
// Amélioration : Lire -> Remove IDs traités -> Write.
// Mais entre le début du script et maintenant, Dashboard a pu ajouter des lignes.
// Re-lecture sécurisée.
// On va faire une logique de mise à jour sécurisée sur le fichier.

$currentQueue = CsvHandler::read(FILE_SEND_QUEUE);
// On filtre ceux qu'on a traités avec succès.
$processedIds = array_column($toProcess, 'id'); // En supposant que ID est unique
// Mais si échec d'envoi, on ne l'a pas mis dans archive, donc on veut le garder.
// Seuls ceux qui ont succes = true sont à virer.
// Je n'ai pas stocké quels IDs ont réussi spécifiquement dans une liste clean, faisons-le.

// Re-logique clean
// Pour l'instant, sender.php est le seul consommateur (remove) de SEND_QUEUE.
// Dashboard est le seul producteur (append).
// Donc CsvHandler::read + remove + overwrite est "safe" SI on lock, mais on a relaché le lock.
// Si Dashboard append pdt le sleep(30), overwrite sans relire écrase l'append.

// Solution:
// 1. Lock EX sur SEND_QUEUE.
// 2. Read tout.
// 3. Identifier les IDs qu'on vient d'envoyer (succès).
// 4. Filtrer.
// 5. Write.
// 6. Unlock.

// Implémentation :
// Ici, on a DEJA envoyé les mails. Donc on doit mettre à jour le fichier maintenant.
$handle = fopen(FILE_SEND_QUEUE, 'c+'); // Read/Write
if (flock($handle, LOCK_EX)) {
    // Lire le contenu actuel à jour
    $fileData = [];
    $headers = fgetcsv($handle);
    while (($row = fgetcsv($handle)) !== false) {
        if ($headers && count($row) == count($headers)) {
            $fileData[] = array_combine($headers, $row);
        }
    }

    // Filtrer
    $newQueue = [];
    foreach ($fileData as $row) {
        $shouldRemove = false;
        foreach ($toProcess as $processedItem) {
            // Comparaison ID. Si envoi réussi pour cet ID, on remove.
            // (La variable $success est locale dans la boucle foreach plus haut, pas dispo ici directement).
            // On a besoin de savoir quels IDs ont été envoyés avec succès.
            // Hack: On n'utilise pas $processedIds générique, on doit tracker $successfulIds.
        }
        // ... (Logique complexe).
    }

    // Simplification pour ce contexte :
    // On suppose que le dashboard ajoute rarement à la seconde près.
    // On va utiliser une méthode 'removeIds(FILE, [ids])' dans CsvHandler serait top.
}
fclose($handle);

// Bref, pour faire simple et lisible dans ce fichier sans refaire CsvHandler:
// On va juste faire : Read à nouveau, filtrer, Overwrite.
// Le risque de race condition existe mais est faible si Dashboard utilise Append (Lock EX).
// Append attendra que Overwrite finisse.
// MAIS si Overwrite ne contient pas les nouvelles lignes de Append... C'est là le risque.
// Dashboard Append -> Lock -> Write -> Unlock.
// Sender Read -> (Memory) -> Dashboard Append (Wait) -> ...
// Sender Overwrite -> Lock -> Write (Old Snapshot filtered) -> Unlock.
// -> Les lignes de Dashboard sont perdues.

// Solution "Atomic Drop" :
// Sender ne devrait pas Overwrite tout le fichier.
// Sender devrait lire, prendre X lignes, et réécrire le fichier en supprimant CES lignes en gardant le reste.
// C'est ce que fait le bloc ci-dessous.

updateSendQueuePostSend($toProcess, $smtp); // Appel fictif pour structurer

Logger::info("--- END SENDER ---");


// --- Helpers ---
function updateSendQueuePostSend($originalSelection, $smtpInstance)
{
    // Cette fonction refait la logique d'envoi et update de manière lockée ? 
    // Non, on ne peut pas locker pendant 30s.

    // Approche Robuste :
    // 1. Lire, Lock.
    // 2. Prendre les 2 premiers.
    // 3. Réécrire le fichier SANS ces 2 premiers (Shifting).
    // 4. Unlock.
    // 5. Envoyer les mails.
    // 6. Si échec, Ré-append en fin de queue (ou log error).

    // C'est beaucoup plus sûr !

    $lockedHandle = fopen(FILE_SEND_QUEUE, 'c+');
    if (!flock($lockedHandle, LOCK_EX)) {
        Logger::error("Impossible de locker la queue pour envoi.");
        return;
    }

    // Lire contenu
    // On rembobine
    rewind($lockedHandle);
    $headers = fgetcsv($lockedHandle);
    $rows = [];
    if ($headers) {
        while (($r = fgetcsv($lockedHandle)) !== false) {
            if (count($r) == count($headers))
                $rows[] = array_combine($headers, $r);
        }
    }

    if (empty($rows)) {
        flock($lockedHandle, LOCK_UN);
        fclose($lockedHandle);
        return;
    }

    // Prendre les 2 premiers
    $batch = array_slice($rows, 0, SMTP_MAX_MAILS_PER_MINUTE);
    $remaining = array_slice($rows, SMTP_MAX_MAILS_PER_MINUTE);

    // Réécrire le fichier avec $remaining TOUT DE SUITE
    ftruncate($lockedHandle, 0);
    rewind($lockedHandle);
    fputcsv($lockedHandle, $headers);
    foreach ($remaining as $r) {
        fputcsv($lockedHandle, $r);
    }
    fflush($lockedHandle);
    flock($lockedHandle, LOCK_UN);
    fclose($lockedHandle);

    // Maintenant on envoie
    $sentCount = 0;
    foreach ($batch as $item) {
        Logger::info("Envoi (Secure) à {$item['email']}");
        $sent = $smtpInstance->send($item['email'], "Re: " . $item['subject'], $item['grok_response']);

        if ($sent) {
            $sentCount++;
            // Archive
            $item['date_sent'] = date('Y-m-d H:i:s');
            $item['final_response'] = $item['grok_response'];
            $archiveRow = [
                'id' => $item['id'],
                'email' => $item['email'],
                'subject' => $item['subject'],
                'date_sent' => $item['date_sent'],
                'tag_zimbra' => $item['tag_zimbra'],
                'final_response' => $item['final_response'],
                'shop_id' => $item['shop_id'] ?? ''
            ];
            CsvHandler::append(FILE_ARCHIVES, $archiveRow);

            if ($sentCount < count($batch))
                sleep(SMTP_PAUSE_BETWEEN_MAILS);
        } else {
            // Echec -> On remet en queue (Append à la fin pour réessayer plus tard)
            Logger::error("Echec envoi, remise en fin de queue.");
            CsvHandler::append(FILE_SEND_QUEUE, $item);
        }
    }
}
