<?php
// core/SpamFilter.php

class SpamFilter
{
    private static $keywords = null;

    /**
     * Load keywords from CSV once per request.
     */
    public static function getKeywords(): array
    {
        if (self::$keywords !== null)
            return self::$keywords;

        self::$keywords = [];
        $file = __DIR__ . '/../data/spam.csv';
        if (file_exists($file)) {
            $handle = fopen($file, 'r');
            if ($handle) {
                fgetcsv($handle); // Headers
                while (($row = fgetcsv($handle)) !== false) {
                    if (!empty($row[0])) {
                        self::$keywords[] = strtolower(trim($row[0]));
                    }
                }
                fclose($handle);
            }
        }
        return self::$keywords;
    }

    /**
     * Check if a row should be classified as Spam (Manual Keywords).
     */
    public static function isSpam(array $item): bool
    {
        $keywords = self::getKeywords();
        if (empty($keywords))
            return false;

        $fields = [
            $item['subject'] ?? '',
            $item['email'] ?? '',
            $item['sender_name'] ?? '',
            $item['body'] ?? '',
            $item['grok_response'] ?? ''
        ];
        $haystack = mb_strtolower(implode(' ', $fields), 'UTF-8');

        foreach ($keywords as $kw) {
            if ($kw !== '' && mb_strpos($haystack, $kw, 0, 'UTF-8') !== false) {
                return true;
            }
        }
        return false;
    }

    /**
     * Determine visibility status for tabs.
     * returns: 'PENDING', 'SPAM', 'SUSPICION'
     */
    public static function getTargetTab(array $item): string
    {
        // Rule 1: Manual Keywords ALWAYS go to Spam
        if (self::isSpam($item) || ($item['status'] ?? '') === 'SPAM_KEYWORD') {
            return 'SPAM';
        }

        // Rule 2: If AI says spam but no keywords, it's Suspicion
        if (($item['status'] ?? '') === 'SPAM_SUSPECT') {
            return 'SUSPICION';
        }

        // Default
        return 'PENDING';
    }
}
